'use strict';

/* --------------------------------------------------------------
   PayPalLoader.js 2023-04-12
   Gambio GmbH
   http://www.gambio.de
   Copyright (c) 2023 Gambio GmbH
   Released under the GNU General Public License (Version 2)
   [http://www.gnu.org/licenses/gpl-2.0.html]
   --------------------------------------------------------------
*/

(function () {
    if (window.NodeList && !NodeList.prototype.forEach) {
        NodeList.prototype.forEach = Array.prototype.forEach;
    }

    var currentScript = document.currentScript;

    var consoleInfo = function consoleInfo() {
        var _console;

        if (typeof payPalButtonSettings.developmentMode !== 'boolean' || payPalButtonSettings.developmentMode === false) {
            return;
        }

        for (var _len = arguments.length, params = Array(_len), _key = 0; _key < _len; _key++) {
            params[_key] = arguments[_key];
        }

        (_console = console).info.apply(_console, ['PayPalLoader'].concat(params));
    };

    var handleAntiqueBrowser = function handleAntiqueBrowser() {
        consoleInfo('Sorry, antique browser not supported.');

        var pppay = document.querySelector('li.gambio_hub-PayPal2Hub');
        if (pppay) {
            pppay.remove();
        }
    };

    var initDisplayModeECS_ButtonReplace = function initDisplayModeECS_ButtonReplace() {
        var paypalButtonContainer = document.querySelector('#paypal-button-container'),
            footerTotalRow = document.querySelector('table.order-total tr.footer.total'),
            newTotalRow = document.createElement('tr'),
            newTotalRowCell = document.createElement('td');
        newTotalRowCell.colSpan = '2';
        newTotalRowCell.style.width = '100%';
        newTotalRowCell.style.borderTop = 'none';
        newTotalRowCell.appendChild(paypalButtonContainer);
        newTotalRow.appendChild(newTotalRowCell);
        footerTotalRow.parentNode.appendChild(newTotalRow);
        footerTotalRow.querySelectorAll('td').forEach(function (td) {
            td.style.paddingBottom = '15px';
        });
    };

    var initDisplayModeECS = function initDisplayModeECS() {
        if (window.location.search.match(/(\?|&)display_mode=ecs($|&)/)) {
            var checkoutButtons = document.querySelector('div.checkout-buttons');
            if (checkoutButtons) {
                checkoutButtons.style.display = 'none';
            }
            var checkoutSubmitButton = document.querySelector('div.shopping-cart-button a.button-submit');
            if (checkoutSubmitButton === null) {
                checkoutSubmitButton = document.querySelector('tr.checkout-button');
            }
            if (checkoutSubmitButton) {
                checkoutSubmitButton.style.display = 'none';
            }
            var shoppingCartButton = document.querySelector('div.shopping-cart-button');
            if (shoppingCartButton) {
                shoppingCartButton.classList.add('paypal-ecs-mode');
            }
            var ppiContainer = document.querySelector('div.paypalinstallmentcontainer');
            if (ppiContainer) {
                ppiContainer.style.display = 'none';
            }
            initDisplayModeECS_ButtonReplace();
        }
    };

    var initJSSDKPayPalButtonECS = function initJSSDKPayPalButtonECS(amount) {
        if (amount === undefined) {
            amount = payPalButtonSettings.cartAmount;
        }
        consoleInfo('initJSSDKPayPalButtonECS cart amount: ' + amount);
        if (amount < 0.01) {
            consoleInfo('ECS: not showing, cart amount too low');
            return;
        }
        var buttonContainer = document.querySelector('#paypal-button-container');
        if (!buttonContainer) {
            return;
        }
        var ecsIntro = document.querySelector('div.ecs_intro');
        if (document.querySelector('tr.checkout-button .button-disabled') !== null) {
            if (ecsIntro) {
                ecsIntro.style.display = 'none';
            }
            buttonContainer.style.display = 'none';
        } else {
            if (ecsIntro) {
                ecsIntro.style.display = 'block';
            }
            buttonContainer.style.display = 'block';
            initDisplayModeECS();
        }

        paypal.Buttons({
            style: payPalButtonSettings.style,
            createOrder: async function createOrder(data, actions) {
                return await createOrderWithOrWithoutVault();
            },
            onApprove: function onApprove(data, actions) {
                consoleInfo('Approved data:', data);
                return fetch(payPalButtonSettings.approvedOrderUrl + '&orderId=' + data.orderID, {
                    method: 'post',
                    headers: {
                        'Content-Type': 'application/json'
                    },
                    body: JSON.stringify(data)
                }).then(function (res) {
                    return res.json();
                }).then(function (responsedata) {
                    consoleInfo('response data:', responsedata);
                    document.location = payPalButtonSettings.checkoutUrl;
                });
            },
            onError: function onError(err) {
                var ppButtonContainer = document.querySelector('#paypal-button-container');
                var ecsIntro = document.querySelector('div.ecs_intro');
                if (ppButtonContainer) {
                    var errorMessage = document.querySelector('div.paypal-error');
                    if (!errorMessage) {
                        errorMessage = document.createElement('div');
                        errorMessage.classList.add('paypal-error');
                        errorMessage.classList.add('alert');
                        errorMessage.classList.add('alert-warning');
                        errorMessage.style.textAlign = 'left';
                        errorMessage.style.color = '#555';
                    }
                    errorMessage.innerText = payPalText.paypalUnavailable + ' ';
                    var shoppingCartButton = document.querySelector('div.shopping-cart-button');
                    if (shoppingCartButton && shoppingCartButton.classList.contains('paypal-ecs-mode')) {
                        var linkUrl = window.location.toString().replace('display_mode=ecs', 'display_mode=normal');
                        var continueLink = document.createElement('a');
                        continueLink.setAttribute('href', linkUrl);
                        continueLink.innerText = payPalText.errorContinue;
                        errorMessage.append(continueLink);
                    }
                    ppButtonContainer.parentNode.append(errorMessage);
                    ppButtonContainer.style.display = 'none';
                }
                if (ecsIntro) {
                    ecsIntro.style.display = 'none';
                }
            }
        }).render('#paypal-button-container');

        var observerTarget = document.querySelector('#paypal-button-container');
        var observer = new MutationObserver(function (mutations, observer) {
            mutations.forEach(function (mutation) {
                if (mutation.removedNodes.length > 0) {
                    consoleInfo('re-init PayPal buttons');
                    var totalSumCell = document.querySelector('tr.total.sum td:nth-child(2)');
                    if (totalSumCell) {
                        var _amount = parseFloat(totalSumCell.textContent.replace(/[^0-9]/g, '')) / 100;
                        initJSSDKPayPalButtonECS(_amount);
                        initInstallmentBanners(_amount);
                    } else {
                        initJSSDKPayPalButtonECS();
                        initInstallmentBanners();
                    }
                }
            });
        });
        observer.observe(observerTarget, { childList: true });
        appendVaultCheckbox(buttonContainer);
    };

    var addButtonContainerDecoration = function addButtonContainerDecoration(buttonContainer, continueButtonBlock) {
        buttonContainer.style.width = 'auto';
        buttonContainer.style.textAlign = 'center';
        buttonContainer.style.fontStyle = 'italic';
        if (payPalText.continueToPayPal) {
            var labelToButtonDistance = 3;
            var lineHeight = '-' + window.getComputedStyle(continueButtonBlock).lineHeight;
            buttonContainer.style.marginTop = 'calc(' + lineHeight + ' - ' + labelToButtonDistance + 'px)';
            var continueLabel = document.createElement('span');
            continueLabel.classList.add('paypal-continue-label');
            continueLabel.textContent = payPalText.continueToPayPal;
            continueLabel.style.paddingBottom = labelToButtonDistance + 'px';
            buttonContainer.appendChild(continueLabel);
        }
    };

    var initJSSDKPayPalButtonECM = function initJSSDKPayPalButtonECM(continueButtonBlock) {
        var paypalButtonContainer = document.createElement('div');

        paypalButtonContainer.id = 'paypal-button-container';
        paypalButtonContainer.style.display = 'none';
        addButtonContainerDecoration(paypalButtonContainer, continueButtonBlock);
        continueButtonBlock.appendChild(paypalButtonContainer);
        paypal.Buttons({
            fundingSource: 'paypal',
            style: payPalButtonSettings.style,
            createOrder: async function createOrder(data, actions) {
                return await createOrderWithOrWithoutVault(payPalButtonSettings.usePayNow);
            },
            onApprove: function onApprove(data, actions) {
                document.querySelector('#checkout_payment input[name="PayPal2HubOrderId"]').value = data.orderID;
                document.querySelector('#checkout_payment input[name="PayPal2HubPayerId"]').value = data.payerID;
                document.querySelector('#checkout_payment input[name="PayPal2HubPaymentInitiator"]').value = 'ecm';
                document.querySelector('#checkout_payment').submit();
            }
        }).render('#paypal-button-container');
        appendVaultCheckbox(paypalButtonContainer);
        return paypalButtonContainer;
    };

    /**
     * Show save payment option.
     * @param paypalButtonContainer
     * @returns {*}
     */
    function appendVaultCheckbox(paypalButtonContainer) {
        if (payPalButtonSettings.useVault !== true) {
            //no enable vault
            return;
        }
        // only enable for shop enable vault and customer has'not using vault yet.
        if (!isCustomerUsingVaultToken()) {
            // Checkbox already exists in the container
            if (paypalButtonContainer.querySelector('#paypal-vaulting-checkbox')) {
                return;
            }
            var labelTexts = {
                "en_US": "Save PayPal account for future payments",
                "de_DE": "Für zukünftige Einkäufe speichern"
            };
            var divTag = document.createElement('div');
            divTag.id = 'paypal-vaulting-div';
            divTag.style.marginTop = '3px';
            var checkbox = document.createElement('input');
            checkbox.type = 'checkbox';
            checkbox.id = 'paypal-vaulting-checkbox';
            checkbox.value = true;
            // Create the label
            var label = document.createElement('label');
            label.htmlFor = 'paypal-vaulting-checkbox';
            label.style.display = 'inline';
            label.style.marginLeft = '3px';
            label.textContent = labelTexts[payPalButtonSettings.locale] || labelTexts["de_DE"];

            // Append the checkbox and label to the DOM (e.g., to the body)
            divTag.appendChild(checkbox);
            divTag.appendChild(label);
            paypalButtonContainer.appendChild(divTag);
        } else {
            appendCustomVaultCss();
        }
    }

    /**
     * Append some customer css class for vault
     */
    function appendCustomVaultCss() {
        try {
            var paypalContainer = document.getElementById('paypal-button-container');
            if (!paypalContainer) return;

            var isShoppingCart = document.querySelector('.shopping-cart-button');
            var isCheckout = document.querySelector('.continue_button');

            if (isShoppingCart) {
                paypalContainer.classList.add('paypal-shopping-cart-vault');
            } else if (isCheckout) {
                paypalContainer.classList.add('paypal-checkout-vault');

                var continueButton = document.querySelector('.continue_button');
                var backButton = document.querySelector('.back_button');
                if (continueButton) {
                    continueButton.classList.add('continue_button_vault');
                }
                if (backButton) {
                    backButton.classList.add('back_button_vault');
                }
            }
        } catch (error) {
            console.error('Error appending custom vault CSS:', error);
        }
    }

    /**
     * Check customer is using vault token
     * @returns {boolean}
     */
    function isCustomerUsingVaultToken() {
        var vaultToken = payPalButtonSettings.vaultToken;

        return vaultToken !== null && typeof vaultToken !== 'undefined' && vaultToken !== "";
    }

    /**
     * Separate a function create order with or without vault for paypal button
     * @returns {Promise<*>}
     */
    async function createOrderWithOrWithoutVault() {
        var payNow = arguments.length > 0 && arguments[0] !== undefined ? arguments[0] : false;

        var url = new URL(payPalButtonSettings.createOrderUrl);
        if (payNow === true || payNow === 'true') {
            url.searchParams.set('payNow', 'true');
        }
        var vaultCheckbox = document.getElementById('paypal-vaulting-checkbox');
        var shouldVault = vaultCheckbox && vaultCheckbox.checked || isCustomerUsingVaultToken();
        if (payPalButtonSettings.useVault === true && shouldVault === true) {
            url.searchParams.set('vault', 'true');
            url.searchParams.set('payment_method', 'paypal');
        }
        var response = await fetch(url.toString(), {
            method: 'POST',
            headers: {
                'Content-Type': 'application/json'
            }
        });
        var data = await response.json();
        return data.id;
    }

    var initPayLaterButton = function initPayLaterButton(continueButtonBlock) {
        var payLaterItem = document.querySelector('li.gambio_hub-PayPal2Hub-paylater');
        if (payLaterItem === null) {
            return null;
        }

        var paylaterButton = paypal.Buttons({
            fundingSource: 'paylater',
            style: payPalButtonSettings.style,
            createOrder: function createOrder(data, actions) {
                return fetch(payPalButtonSettings.createOrderUrl, {
                    method: 'post',
                    headers: {
                        'Content-Type': 'application/json'
                    }
                }).then(function (res) {
                    return res.json();
                }).then(function (orderdata) {
                    consoleInfo('order created: ' + orderdata.id, orderdata);
                    return orderdata.id;
                });
            },
            onApprove: function onApprove(data, actions) {
                document.querySelector('#checkout_payment input[name="PayPal2HubOrderId"]').value = data.orderID;
                document.querySelector('#checkout_payment input[name="PayPal2HubPayerId"]').value = data.payerID;
                document.querySelector('#checkout_payment').submit();
            }
        });

        if (paylaterButton.isEligible()) {
            var paylaterButtonId = 'paypal-paylater-button-container',
                paylaterButtonContainer = document.createElement('div');
            paylaterButtonContainer.id = paylaterButtonId;
            paylaterButtonContainer.style.display = 'none';
            addButtonContainerDecoration(paylaterButtonContainer, continueButtonBlock);
            continueButtonBlock.appendChild(paylaterButtonContainer);
            paylaterButton.render('#' + paylaterButtonId);
            consoleInfo('PayPal Paylater: eligible and initialized');
            return paylaterButtonContainer;
        } else {
            consoleInfo('PayPal Paylater: not eligible');
            payLaterItem.remove();
        }

        return null;
    };

    var initSepaButton = function initSepaButton(continueButtonBlock) {
        var sepaItem = document.querySelector('li.gambio_hub-PayPal2Hub-sepa');
        if (sepaItem === null) {
            return null;
        }
        var sepaButtonStyle = payPalButtonSettings.style;
        if (sepaButtonStyle.color === 'gold' || sepaButtonStyle.color === 'blue') {
            sepaButtonStyle.color = 'silver';
        }

        var sepaButton = paypal.Buttons({
            fundingSource: 'sepa',
            style: payPalButtonSettings.style,
            createOrder: function createOrder(data, actions) {
                return fetch(payPalButtonSettings.createOrderUrl, {
                    method: 'post',
                    headers: {
                        'Content-Type': 'application/json'
                    }
                }).then(function (res) {
                    return res.json();
                }).then(function (orderdata) {
                    consoleInfo('order created: ' + orderdata.id, orderdata);
                    return orderdata.id;
                });
            },
            onApprove: function onApprove(data, actions) {
                document.querySelector('#checkout_payment input[name="PayPal2HubOrderId"]').value = data.orderID;
                document.querySelector('#checkout_payment input[name="PayPal2HubPayerId"]').value = data.payerID;
                document.querySelector('#checkout_payment').submit();
            }
        });

        if (sepaButton.isEligible()) {
            var sepaButtonId = 'paypal-sepa-button-container',
                sepaButtonContainer = document.createElement('div');
            sepaButtonContainer.id = sepaButtonId;
            sepaButtonContainer.style.display = 'none';
            addButtonContainerDecoration(sepaButtonContainer, continueButtonBlock);
            continueButtonBlock.appendChild(sepaButtonContainer);
            sepaButton.render('#' + sepaButtonId);
            consoleInfo('PayPal SEPA: eligible and initialized');
            return sepaButtonContainer;
        } else {
            consoleInfo('PayPal SEPA: not eligible');
            sepaItem.remove();
        }

        return null;
    };

    var initCreditCardButton = function initCreditCardButton(continueButtonBlock) {
        var brandedCreditCardsItem = document.querySelector('li.gambio_hub-PayPal2Hub-creditcardbutton');
        if (brandedCreditCardsItem === null) {
            return null;
        }

        var cardButtonStyle = payPalButtonSettings.style;
        cardButtonStyle.color = 'black';
        cardButtonStyle.shape = 'pill';

        var creditCardButton = paypal.Buttons({
            fundingSource: 'card',
            style: cardButtonStyle,
            createOrder: function createOrder(data, actions) {
                return fetch(payPalButtonSettings.createOrderUrl, {
                    method: 'post',
                    headers: {
                        'Content-Type': 'application/json'
                    }
                }).then(function (res) {
                    return res.json();
                }).then(function (orderdata) {
                    consoleInfo('order created: ' + orderdata.id, orderdata);
                    return orderdata.id;
                });
            },
            onApprove: function onApprove(data, actions) {
                document.querySelector('#checkout_payment input[name="PayPal2HubOrderId"]').value = data.orderID;
                document.querySelector('#checkout_payment input[name="PayPal2HubPayerId"]').value = data.payerID;
                document.querySelector('#checkout_payment').submit();
            }
        });

        consoleInfo(creditCardButton);

        if (creditCardButton.isEligible()) {
            consoleInfo('Eligible for branded cc payments');
            var creditCardButtonId = 'paypal-creditcard-button-container',
                creditCardButtonContainer = document.createElement('div');
            creditCardButtonContainer.id = creditCardButtonId;
            creditCardButtonContainer.style.display = 'none';
            addButtonContainerDecoration(creditCardButtonContainer, continueButtonBlock);
            continueButtonBlock.appendChild(creditCardButtonContainer);
            creditCardButton.render('#' + creditCardButtonId);
            return creditCardButtonContainer;
        } else {
            consoleInfo('NOT eligible for branded cc payments');
            brandedCreditCardsItem.remove();
        }

        return null;
    };

    var initHostedCreditCard = function initHostedCreditCard() {
        var theLabel = document.querySelector('#ppcc-card-holder-field').closest('label'),
            theDiv = document.createElement('div');

        theDiv.classList.add('payment-module-container');
        theLabel.parentNode.appendChild(theDiv);
        while (theLabel.hasChildNodes()) {
            theDiv.appendChild(theLabel.firstChild);
        }
        theLabel.remove();

        var theListItem = document.querySelector('li.gambio_hub-PayPal2Hub-creditcard');
        theListItem.style.cursor = 'pointer';
        theListItem.addEventListener('click', function () {
            $(this).find('input:radio:not(:disabled):not(.placeholder-radio)').first().prop('checked', true).trigger('change');
        });

        var cardHolderField = document.querySelector('#ppcc-card-holder-field'),
            fieldStyle = getComputedStyle(cardHolderField),
            orderIdElement = document.querySelector('#checkout_payment input[name="PayPal2HubOrderId"]');

        paypal.HostedFields.render({
            createOrder: function createOrder() {
                return fetch(payPalButtonSettings.createOrderUrl, {
                    method: 'post',
                    headers: {
                        'Content-Type': 'application/json'
                    }
                }).then(function (res) {
                    return res.json();
                }).then(function (orderdata) {
                    orderIdElement.value = orderdata.id;
                    consoleInfo('credit card order created: ' + orderdata.id, orderdata);
                    return orderdata.id;
                });
            },
            styles: {
                'input': {
                    'color': fieldStyle.color,
                    'font-size': fieldStyle.fontSize,
                    'font-family': fieldStyle.fontFamily,
                    'padding': fieldStyle.padding,
                    'line-height': fieldStyle.lineHeight
                },
                '.valid': {
                    'color': 'green'
                },
                '.invalid': {
                    'color': 'red'
                }
            },
            fields: {
                number: {
                    'selector': '#ppcc-card-number',
                    'placeholder': '4111111111111111'
                },
                cvv: {
                    'selector': '#ppcc-cvv',
                    'placeholder': '123'
                },
                expirationDate: {
                    'selector': '#ppcc-expiration-date',
                    'placeholder': 'MM/YY'
                }
            }
        }).then(function (cardFields) {
            consoleInfo('PayPal: CC fields initialized', cardFields);
            var paymentForm = document.querySelector('#checkout_payment');
            var cardFieldsSubmitted = false;
            paymentForm.addEventListener('submit', function (event) {
                if (cardFieldsSubmitted === false && paymentForm.payment.value === 'gambio_hub-PayPal2Hub-creditcard') {
                    event.preventDefault();
                    var billingAddressData = JSON.parse(document.querySelector('#ppcc-billingaddress').textContent);
                    cardFields.submit({
                        cardholderName: paymentForm.ppcc_card_holder.value,
                        billingAddress: billingAddressData,
                        contingencies: ['SCA_WHEN_REQUIRED']
                    }).then(function () {
                        // OK
                        consoleInfo('cardFields submitted');
                        cardFieldsSubmitted = true;
                        paymentForm.submit();
                    }).catch(function (err) {
                        orderIdElement.value = '';
                        consoleInfo(err);
                        alert(payPalText.errorCheckData);
                        document.querySelector('input[name="payment"]:checked').scrollIntoView(false);
                    });
                }
            });
        });
    };
    var initLoadingButton = function initLoadingButton(continueButtonBlock) {
        var loadingButtonId = 'loading-button-container';
        var loadingButtonContainer = document.createElement('div');
        loadingButtonContainer.id = loadingButtonId;
        loadingButtonContainer.style.display = 'none';
        loadingButtonContainer.style.width = 'auto';
        loadingButtonContainer.style.verticalAlign = 'inherit';
        loadingButtonContainer.style.backgroundImage = "url(https://cdn.gambiohub.com/sdk/images/loading.gif)";
        loadingButtonContainer.style.backgroundSize = "contain";
        loadingButtonContainer.style.backgroundRepeat = "no-repeat";
        loadingButtonContainer.style.backgroundPosition = "center";
        var parentHeight = continueButtonBlock.clientHeight;
        loadingButtonContainer.style.height = parentHeight + 'px';
        window.addEventListener('showLoadingButton', function (evt) {
            loadingButtonContainer.style.display = 'block';
        });
        window.addEventListener('hideLoadingButton', function (evt) {
            loadingButtonContainer.style.display = 'none';
        });
        continueButtonBlock.appendChild(loadingButtonContainer);
    };

    var initAlternativePaymentMethod = function initAlternativePaymentMethod() {
        try {
            var loader = new PayPalAlternativePaymentMethodLoader.PayPalAlternativePaymentMethodLoader();
            return loader.load();
        } catch (err) {
            consoleInfo(err);
            return [];
        }
    };

    var initCheckoutPayment = function initCheckoutPayment() {
        var continueButtonBlock = document.querySelector('#checkout_payment div.continue_button'),
            continueButton = continueButtonBlock.querySelector('input[type="submit"]'),
            continueButtonDisplay = continueButton.style.display,
            paypalButtonContainer = document.createElement('div'),
            plusContainer = document.querySelector('#gambiohub-ppplus'),
            ccForm = document.querySelector('div.paypal-cc-form'),
            orderIdElement = document.querySelector('#checkout_payment input[name="PayPal2HubOrderId"]'),
            ccCardHolderFieldForm = document.querySelector('#paypal-cardfields-form-container');

        if (plusContainer === null && ccForm !== null && ccCardHolderFieldForm === null) {
            consoleInfo('PayPal: Credit Card form found on page.');
            var hostedFieldsEligible = paypal.HostedFields.isEligible();
            var brandedCreditCardsOption = document.querySelector('li.gambio_hub-PayPal2Hub-creditcardbutton');
            if (hostedFieldsEligible) {
                consoleInfo('PayPal: eligible for hosted fields');
                if (brandedCreditCardsOption) {
                    brandedCreditCardsOption.remove();
                }

                initHostedCreditCard();
            } else {
                consoleInfo('PayPal: NOT eligible for hosted fields');
                var ccListItem = document.querySelector('li.gambio_hub-PayPal2Hub-creditcard');
                if (ccListItem) {
                    ccListItem.remove();
                }
            }
        }

        if (plusContainer === null && !payPalButtonSettings.paymentApproved) {
            var backButtonBlock = continueButtonBlock.parentElement.querySelector('div.back_button');
            if (backButtonBlock !== null && backButtonBlock.classList.contains('col-xs-6')) {
                backButtonBlock.classList.remove('col-xs-6');
                backButtonBlock.classList.add('col-xs-4');
                continueButtonBlock.classList.remove('col-xs-6');
                continueButtonBlock.classList.add('col-xs-8');
                var backButton = backButtonBlock.querySelector('a.btn');
                var height = parseInt(getComputedStyle(backButton).height);
                if (isNaN(height)) {
                    height = 25; // fallback
                }

                if (height < 25) height = 25;
                if (height > 50) height = 50;
                payPalButtonSettings.style.height = height;
            }
            var promises = initAlternativePaymentMethod();
            initJSSDKPayPalButtonECM(continueButtonBlock);
            initPayLaterButton(continueButtonBlock);
            initCreditCardButton(continueButtonBlock);
            initSepaButton(continueButtonBlock);
            initLoadingButton(continueButtonBlock);
            var payPalContainerButtonConfig = [{
                'value': 'gambio_hub-PayPal2Hub',
                'container': 'paypal-button-container'
            }, {
                'value': 'gambio_hub-PayPal2Hub-paylater',
                'container': 'paypal-paylater-button-container'
            }, {
                'value': 'gambio_hub-PayPal2Hub-sepa',
                'container': 'paypal-sepa-button-container'
            }, {
                'value': 'gambio_hub-PayPal2Hub-creditcardbutton',
                'container': 'paypal-creditcard-button-container'
            }, {
                'value': 'loading-button-container',
                'container': 'loading-button-container'
            }];
            var paymentItemClickListener = function paymentItemClickListener() {
                var selected_payment = this.querySelector('input[name="payment"]');
                if (null !== selected_payment) {
                    var showDefaultButton = true;
                    var selectedPaymentValue = selected_payment.value;
                    if (selectedPaymentValue === 'gambio_hub' && selected_payment.dataset.module_code === 'PayPal2Hub') {
                        selectedPaymentValue = 'gambio_hub-PayPal2Hub';
                    }
                    payPalContainerButtonConfig.forEach(function (item) {
                        var button = document.getElementById(item.container);
                        if (button) {
                            if (item.value === selectedPaymentValue) {
                                showDefaultButton = false;
                                button.style.display = 'block';
                                continueButton.style.display = 'none';
                            } else {
                                button.style.display = 'none';
                            }
                        }
                    });
                    if (showDefaultButton === true) {
                        continueButton.style.display = continueButtonDisplay;
                    }
                }
            };

            Promise.allSettled(promises).then(function (results) {
                results.forEach(function (result) {
                    if (result.status === 'fulfilled' && result.value) {
                        payPalContainerButtonConfig.push(result.value);
                    }
                });
                var paymentItems = document.querySelectorAll('#checkout_payment input[name="payment"], #checkout_payment li.list-group-item');
                paymentItems.forEach(function (paymentItem) {
                    paymentItem.addEventListener('click', paymentItemClickListener);
                });

                var paymentListEntries = document.querySelectorAll('#checkout_payment li');
                paymentListEntries.forEach(function (paymentOption) {
                    if (paymentOption.querySelector('input[name="payment"]:checked') || paymentListEntries.length === 1) {
                        paymentOption.dispatchEvent(new Event('click'));
                    }
                });
            });
        }
    };

    /****
     ****  Installment Banners
    	 */

    var initInstallmentBanners = function initInstallmentBanners(amount) {
        consoleInfo('Initialising PayPal Installments banners', 'CartAmount = ' + payPalBannerSettings.cartAmount);
        var banners = document.querySelectorAll('.paypal-installments');
        banners.forEach(function (bannerElement) {
            var position = bannerElement.dataset.ppinstPos;
            consoleInfo('found banner on position ' + position);
            if (payPalBannerSettings.positions && payPalBannerSettings.positions[position]) {
                consoleInfo('settings found for position ' + position, payPalBannerSettings.positions[position]);
                if (payPalBannerSettings.positions[position].style.layout === 'none') {
                    consoleInfo('position disabled: ' + position);
                    return;
                }

                if (amount === undefined) {
                    amount = 0.00;
                }
                if (payPalBannerSettings.cartAmount) {
                    amount += payPalBannerSettings.cartAmount;
                }
                if (payPalBannerSettings.productsPrice) {
                    amount += payPalBannerSettings.productsPrice;
                } else {
                    var p = document.querySelector('div.current-price-container');
                    if (p) {
                        var priceText = p.innerText;
                        priceText = priceText.replace(/.*?[\d,.]+\s+\D{1,3}.*?([\d,.]+\s+\D{1,3})/s, '$1');
                        amount += 0.01 * parseFloat(priceText.replace(/.*?(((\d{1,3}[.,])+)(\d{2})).*/, '$1').replace(/[.,]/g, ''));
                        consoleInfo('Product amount for banner: ' + amount);
                    }
                }
                if (amount < 99) {
                    consoleInfo('Not showing PayPal Installments banner for amount ' + amount);
                    return;
                }

                if (bannerElement.classList.contains('paypal-installments-cartbottom')) {
                    var observerTarget = document.querySelector('div.shopping-cart-button');
                    if (observerTarget) {
                        var cartSumElement = document.querySelector('tr.total.sum td:nth-child(2)');
                        var amountString = '0';
                        if (cartSumElement) {
                            amountString = cartSumElement.textContent.trim();
                        } else {
                            cartSumElement = document.querySelector('tr.footer.total td:nth-child(2)');
                            if (cartSumElement) {
                                amountString = cartSumElement.textContent.trim().replace(/(\n|\t|\.|\,)/g, '').replace(/.*?([0-9.,]+)\s+EUR.*/, '$1');
                            }
                        }
                        amount = 0.01 * parseInt(amountString.replace(/[^0-9]/g, ''));
                        consoleInfo('cart amount ' + amount);

                        var observer = new MutationObserver(function (mutations, observer) {
                            mutations.forEach(function (mutation) {
                                if (mutation.removedNodes.length > 0) {
                                    consoleInfo('re-init PayPal installments banner');
                                    initInstallmentBanners();
                                }
                            });
                        });
                        observer.observe(observerTarget, { childList: true });
                    }
                }

                paypal.Messages({
                    amount: amount,
                    currency: payPalBannerSettings.currency,
                    style: payPalBannerSettings.positions[position].style,
                    placement: payPalBannerSettings.positions[position].placement
                }).render(bannerElement).then(function () {
                    var legacyInstallmentContainer = document.querySelector('.paypalinstallmentcontainer');
                    if (legacyInstallmentContainer) {
                        legacyInstallmentContainer.remove();
                    }
                });
            }
        });
    };

    var paypalSdkLoaded = function paypalSdkLoaded() {
        consoleInfo('PayPalSDK loaded');
        if (window.location.pathname.match(/shopping_cart.php/)) {
            consoleInfo('Initializing ECS button');
            initJSSDKPayPalButtonECS();
        }
        if (window.location.pathname.match(/checkout_payment.php/)) {
            consoleInfo('Initializing PayPal on payment page');
            initCheckoutPayment();
        }
        initInstallmentBanners();
    };

    var disableJavascriptFeatures = function disableJavascriptFeatures() {
        var disable = arguments.length > 0 && arguments[0] !== undefined ? arguments[0] : true;

        /*
        let pppay = document.querySelector('ul:not(.paypal3-plus-checkout) li.gambio_hub-PayPal2Hub');
        if (pppay) {
            consoleInfo('Removing PayPal payment option');
            pppay.remove();
        }
        */

        var newStyle = disable ? 'none' : 'block';

        var ecsButton = document.querySelector('div#paypal-newbutton');
        if (ecsButton) {
            ecsButton.style.display = newStyle;
        }

        var ccPaymentOption = document.querySelector('#checkout_payment li.gambio_hub-PayPal2Hub-creditcard');
        if (ccPaymentOption) {
            ccPaymentOption.style.display = newStyle;
        }

        var puiPaymentOption = document.querySelector('#checkout_payment li.gambio_hub-PayPal2Hub-pui');
        if (puiPaymentOption) {
            puiPaymentOption.style.display = newStyle;
        }

        var paylaterPaymentOption = document.querySelector('#checkout_payment li.gambio_hub-PayPal2Hub-paylater');
        if (paylaterPaymentOption) {
            paylaterPaymentOption.style.display = newStyle;
        }

        var sepaPaymentOption = document.querySelector('#checkout_payment li.gambio_hub-PayPal2Hub-sepa');
        if (sepaPaymentOption) {
            sepaPaymentOption.style.display = newStyle;
        }

        var ecmPaymentOption = document.querySelector('#checkout_payment li.gambio_hub-PayPal2Hub');
        if (ecmPaymentOption && disable) {
            var paypalModeInput = document.createElement('input');
            paypalModeInput.type = 'hidden';
            paypalModeInput.name = 'PayPalMode';
            paypalModeInput.value = 'sdkless';
            ecmPaymentOption.appendChild(paypalModeInput);
        }
    };

    var disablePayPalOptions = function disablePayPalOptions() {
        var disable = arguments.length > 0 && arguments[0] !== undefined ? arguments[0] : true;

        var paymentOptions = document.querySelectorAll('#checkout_payment li.list-group-item');
        paymentOptions.forEach(function (paymentOption, key, parent) {
            paymentOption.classList.forEach(function (className) {
                if (className.match(/.*PayPal2Hub.*/)) {
                    if (disable) {
                        paymentOption.querySelector('input[name="payment"]').setAttribute('disabled', 'disabled');
                        paymentOption.style.opacity = '0.5';
                    } else {
                        paymentOption.querySelector('input[name="payment"]').removeAttribute('disabled');
                        paymentOption.style.opacity = '1';
                    }
                }
            });
        });
    };

    var enablePayPalOptions = function enablePayPalOptions() {
        disablePayPalOptions(false);
    };

    var readyCallback = function readyCallback() {
        consoleInfo('readyCallback start');
        var jssrc = '';
        if (typeof payPalButtonSettings !== 'undefined' && payPalButtonSettings.jssrc) {
            jssrc = payPalButtonSettings.jssrc;
        } else if (typeof payPalBannerSettings !== 'undefined' && payPalBannerSettings.jssrc) {
            jssrc = payPalBannerSettings.jssrc;
        }
        var pageHasBanners = document.querySelectorAll('.paypal-installments').length > 0;
        var isCheckoutPage = window.location.pathname.match(/shopping_cart.php/) || window.location.pathname.match(/checkout_payment.php/);

        var consentGiven = undefined;
        var purposeId = null;
        var consentDataElement = document.getElementById('paypalconsent');
        if (consentDataElement !== null) {
            var consentData = JSON.parse(consentDataElement.textContent);
            purposeId = consentData.purpose_id;
            var gxConsentsString = document.cookie.split('; ').find(function (row) {
                return row.startsWith('GXConsents=');
            });
            var gxConsents = gxConsentsString ? gxConsentsString.split('=')[1] : null;
            if (gxConsents) {
                var gxConsentsData = JSON.parse(gxConsents);
                consentGiven = gxConsentsData.purposeConsents['' + purposeId];
            }
        }

        if (jssrc && (pageHasBanners || isCheckoutPage) && typeof window.paypal === 'undefined') {
            var loadTimeout = null;

            var onPayPalSdkLoad = function onPayPalSdkLoad() {
                clearTimeout(loadTimeout);
                paypalSdkLoaded();
                disableJavascriptFeatures(false);
                enablePayPalOptions();
            };
            var onPayPalSdkError = function onPayPalSdkError() {
                consoleInfo('ERROR loading PayPal Javascript - blocked?');
                consoleInfo('PayPal JSSDK unavailable, disabling features');
                disableJavascriptFeatures();
                enablePayPalOptions();
            };
            window.initPayPalJSSDK = function () {
                if (typeof window.paypal !== 'undefined') {
                    return;
                }
                var jssdk = document.createElement('script');
                jssdk.src = payPalButtonSettings.jssrc ? payPalButtonSettings.jssrc : payPalBannerSettings.jssrc;
                if (payPalButtonSettings.useVault === true && isCustomerUsingVaultToken()) {
                    jssdk.setAttribute('data-user-id-token', payPalButtonSettings.vaultToken);
                }
                if (payPalButtonSettings.partnerAttributionId !== null && payPalButtonSettings.partnerAttributionId !== undefined) {
                    jssdk.setAttribute('data-partner-attribution-id', payPalButtonSettings.partnerAttributionId);
                }
                var acdcIdentity = document.querySelector('#paypal_cc_identity');
                if (acdcIdentity !== null) {
                    var clientIdentity = JSON.parse(acdcIdentity.textContent);
                    if (typeof clientIdentity.client_token !== 'undefined') {
                        jssdk.dataset.clientToken = clientIdentity.client_token;
                    }
                }
                jssdk.onerror = onPayPalSdkError;
                jssdk.onload = onPayPalSdkLoad;

                if (document.currentScript) {
                    document.currentScript.parentNode.insertBefore(jssdk, document.currentScript);
                } else {
                    document.querySelector('script:last-of-type').parentNode.appendChild(jssdk);
                }
            };

            disablePayPalOptions();
            if (purposeId !== null) {
                consoleInfo("External PayPal Javascript is managed by CookieConsent, purpose " + purposeId);
                if (window.PayPalAsOilClearToLoad) {
                    consoleInfo('OIL has cleared PayPal to load');
                    window.initPayPalJSSDK();
                } else {
                    consoleInfo('OIL has not yet cleared PayPal to load');
                    disableJavascriptFeatures();
                    window.addEventListener('PayPalClearedToLoad', function (evt) {
                        consoleInfo('OIL has cleared PayPal to load (Event)');
                        window.initPayPalJSSDK();
                    });
                    if (isCheckoutPage) {
                        consoleInfo('on checkout page; waiting 5s for OIL');
                        loadTimeout = setTimeout(onPayPalSdkError, 5000);
                    }
                }
            } else {
                console.info("Not using CookieConsent integration for PayPal, loading directly");
                window.initPayPalJSSDK();
            }
        } else {
            consoleInfo('PayPal JSSDK unavailable or not required, disabling features');
            disableJavascriptFeatures();
        }
        consoleInfo('readyCallback end');
    };

    var mainCallback = "fetch" in window ? readyCallback : handleAntiqueBrowser;

    if (document.readyState === 'complete' || document.readyState !== 'loading' && !document.documentElement.doScroll) {
        consoleInfo('Readystate complete');
        mainCallback();
    } else {
        consoleInfo('Adding EventListener for DOMContentLoaded');
        document.addEventListener('DOMContentLoaded', mainCallback);
    }
})();
//# sourceMappingURL=data:application/json;charset=utf8;base64,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
